/*
 * XFCE plugin for the MOC player
 * 
 * Control the MOC (Media On Console) player from the XFCE panel!
 * 
 * Author: Jakub Krauz
 * E-mail: jakub.krauz@volny.cz
 * Univerzity of West Bohemia
 * Department of Computer Science and Engineering
 * Pilsen, 2012
 */

#include <string.h>
#include <gtk/gtk.h>
#include <libxfce4ui/libxfce4ui.h>
#include <libxfce4panel/xfce-panel-plugin.h>
#include <glib/gslist.h>
#include <libxfcegui4/xfce_aboutdialog.h>

#include "plugin.h"
#include "plugin_dialogs.h"
#include "logger.h"



/* function prototypes */
static void configure_response(GtkWidget *dialog, gint response, gpointer data);
static void about_response(GtkWidget* dialog, gint response, gpointer data);


/**
 * Handles the "toggled" event of the checkbutton.
 * If the button is active, all control buttons of the plugin are shown in the panel.
 * Otherwise, only one button is shown in the panel. It shows menu with control buttons on click.
 * @param bt The checkbutton.
 * @param data Pointer to the MocPlugin structure.
 */
void
checkbt_toggled(GtkToggleButton* bt, gpointer data)
{
  MocPlugin *moc_plugin = (MocPlugin *) data;
  if (gtk_toggle_button_get_active(bt))
  {
    moc_plugin->buttons_in_panel = TRUE;
    gtk_widget_destroy(moc_plugin->menu_box); // destroy the single panel button
    create_control_widgets(moc_plugin); // create all panel buttons
  }
  else
  {
    moc_plugin->buttons_in_panel = FALSE;
    gtk_widget_destroy(moc_plugin->bt_box); // destroy panel buttons
    create_control_widgets(moc_plugin); // cretae the single button
  }
}


/**
 * Handles the "toggled" event of the radiobuttons.
 * Changes the colour of icons in the xfce panel according to the active radiobutton.
 * @param bt The radiobutton.
 * @param data Pointer to the MocPlugin structure.
 */
void
radiobt_toggled(GtkToggleButton* bt, gpointer data)
{
  const char *label;
  MocPlugin *moc_plugin = (MocPlugin *) data;
  if (gtk_toggle_button_get_active(bt))
  {
    logit(DBG, "Radio button \"%s\" activated", gtk_button_get_label(GTK_BUTTON(bt)));
    label = gtk_button_get_label(GTK_BUTTON(bt));
    snprintf(moc_plugin->icon_color, MAX_COLOR_LEN, "%s", label); // store the chosen colour
    if (GTK_IS_WIDGET(moc_plugin->bt_box))
      gtk_widget_destroy(moc_plugin->bt_box);
    if (GTK_IS_WIDGET(moc_plugin->menu_box))
      gtk_widget_destroy(moc_plugin->menu_box);
    create_control_widgets(moc_plugin); // create new widgets with the requested icon colour
  }
}


/**
 * Create and show the "configure" dialog.
 * The dialog enables user to choose the icon colour (white or black theme) and whether
 * to show all control buttons in the xfce panel or only a single button with a connected menu.
 * @param plugin The XfcePanelPlugin structure.
 * @param moc_plugin The MocPlugin structure.
 */
void
moc_plugin_configure(XfcePanelPlugin* plugin, MocPlugin* moc_plugin)
{
  GtkWidget *dialog;
  GtkWidget *checkbt;
  GtkWidget *rbwhite;
  GtkWidget *rbblack;
  GtkWidget *label;
  GtkWidget *rbbox;
  GSList *rbgroup;
  gboolean white_icons;

  xfce_panel_plugin_block_menu(plugin); // block the plugin menu
  dialog = xfce_titled_dialog_new_with_buttons("MOC Plugin",
                                               GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
                                               GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
                                               GTK_STOCK_CLOSE, GTK_RESPONSE_OK, NULL);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER); // center the dialog on the screen
  gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfce4-settings"); // set the dialog icon

  /* create and show the checkbutton */
  checkbt = gtk_check_button_new_with_label("show all buttons in the panel");
  GTK_TOGGLE_BUTTON(checkbt)->active = moc_plugin->buttons_in_panel;
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), checkbt, TRUE, TRUE, 10);
  g_signal_connect(checkbt, "toggled", G_CALLBACK(checkbt_toggled), (gpointer) moc_plugin);
  gtk_widget_show(checkbt);

  /* create and show the radio buttons */
  rbbox = gtk_vbox_new(TRUE, 5);
  label = gtk_label_new("Choose icon colour:");
  gtk_misc_set_alignment(GTK_MISC(label), 0.0f, 0.5f);
  gtk_box_pack_start(GTK_BOX(rbbox), label, TRUE, TRUE, 0);
  white_icons = !strcmp(moc_plugin->icon_color, WHITE_STR);
  rbwhite = gtk_radio_button_new_with_label(NULL, WHITE_STR);
  rbgroup = gtk_radio_button_get_group(GTK_RADIO_BUTTON(rbwhite));
  gtk_box_pack_start(GTK_BOX(rbbox), rbwhite, TRUE, TRUE, 0);
  rbblack = gtk_radio_button_new_with_label(rbgroup, BLACK_STR);
  gtk_box_pack_start(GTK_BOX(rbbox), rbblack, TRUE, TRUE, 0);
  if (white_icons)
  {
    GTK_TOGGLE_BUTTON(rbwhite)->active = TRUE;
    GTK_TOGGLE_BUTTON(rbblack)->active = FALSE;
  }
  else
  {
    GTK_TOGGLE_BUTTON(rbwhite)->active = FALSE;
    GTK_TOGGLE_BUTTON(rbblack)->active = TRUE;
  }
  g_signal_connect(rbwhite, "toggled", G_CALLBACK(radiobt_toggled), (gpointer) moc_plugin);
  g_signal_connect(rbblack, "toggled", G_CALLBACK(radiobt_toggled), (gpointer) moc_plugin);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), rbbox, TRUE, TRUE, 10);
  gtk_widget_show_all(rbbox);

  /* link the dialog to the plugin, so we can destroy it when the plugin
   * is closed, but the dialog is still open */
  g_object_set_data(G_OBJECT(plugin), "configure-dialog", dialog);

  /* connect the reponse signal to the dialog */
  g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(configure_response), moc_plugin);

  gtk_widget_show(dialog);
}


/**
 * Create and show the "about" dialog. Displays some basic information about the program.
 * @param plugin The XfcePanelPlugin structure.
 */
void moc_plugin_about(XfcePanelPlugin* plugin, MocPlugin* moc_plugin)
{
  GtkWidget *dialog;
  GtkWidget *label;
  
  dialog = xfce_titled_dialog_new_with_buttons("MOC Plugin",
                                               GTK_WINDOW(gtk_widget_get_toplevel(GTK_WIDGET(plugin))),
                                               GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
                                               GTK_STOCK_CLOSE, GTK_RESPONSE_OK, NULL);
  gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER); // center the dialog on the screen
  gtk_window_set_icon_name(GTK_WINDOW(dialog), "xfce4-moc-plugin-black");
  
  
  label = gtk_label_new("xfce4-moc-plugin 1.0");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 10);
  label = gtk_label_new("Control the MOC player from the XFCE panel.");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 10);
  label = gtk_label_new(XFCE_COPYRIGHT_TEXT("2012", "Jakub Krauz"));
  gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label, TRUE, TRUE, 10);
  
  /* link the dialog to the plugin, so we can destroy it when the plugin
   * is closed, but the dialog is still open */
  g_object_set_data(G_OBJECT(plugin), "about-dialog", dialog);
  
  // connect the reponse signal to the dialog
  g_signal_connect(G_OBJECT(dialog), "response", G_CALLBACK(about_response), moc_plugin);
  
  gtk_widget_show_all(dialog);
}


/**
 * Handle the "response" event of the configure-dialog.
 * Because the dialog has only the "close" button, this function only saves plugin settings
 * into the configuration file and destroys the dialog.
 * @param dialog The dialog widget.
 * @param response The response ID.
 * @param data Pointer to the MocPlugin structure.
 */
static void
configure_response(GtkWidget* dialog, gint response, gpointer data)
{  
  MocPlugin *moc_plugin = (MocPlugin *) data;
  
  /* remove the dialog data from the plugin */
  g_object_set_data(G_OBJECT(moc_plugin->plugin), "configure-dialog", NULL);
  
  xfce_panel_plugin_unblock_menu(moc_plugin->plugin);  // unblock the panel menu
  save_settings(moc_plugin);  // save the plugin settings
  gtk_widget_destroy(dialog);  // destroy the dialog
}


/**
 * Handles the "response" event of the about-dialog.
 * Because the dialog has only the "close" button, this function only destroys the dialog.
 * @param dialog
 * @param response
 * @param data
 */
static void
about_response(GtkWidget* dialog, gint response, gpointer data)
{  
  MocPlugin *moc_plugin = (MocPlugin *) data;
  
  /* remove the dialog data from the plugin */
  g_object_set_data(G_OBJECT(moc_plugin->plugin), "about-dialog", NULL);

  gtk_widget_destroy(dialog);  // destroy the dialog
}
